//------------------------------------------------------------------------
// RemoteSelfBalancing
// 倒立振子ロボット
//
//  DMP使用とPIDライブラリ使用版
//  2017/2/13 更新
//
//　シリアルポートは　115200bpsに設定
//
//  注意：センサーが上手く起動しない場合(データ通信ミス)があります(ロボットのモーターが動き出さない)。
//　その場合は何回かリセットをおして再起動を行ってください。
//  使用電源によってPIDのゲインが変わってくるので、使用する電源によって変更してください。
//
//-----------------------------------------------------------------------
// Copyright (C) Future Robotics Technology Center All Right Reserved
//-----------------------------------------------------------------------

#include <PID_v1.h>
#include <TimerOne.h>
#include "LMotorController.h"
#include "I2Cdev.h"
#include <PS2X_lib.h>
#include "MPU6050_6Axis_MotionApps20R.h"
#include "Wire.h"

//#define MANUAL_PID_TUNING  //比例、積分、微分項の調整　(下記と排他)
#define MANUAL_ZERO_TUNING   //ゼロ点調整と　微分項の調整(D) (上記と排他)


//アルカリ電池の場合(使用する電源によって数字を書き直してください↓↓↓)
/**************　変更点　************************/
#define GAIN_KP 39.0         //変更はプロフェッショナルコース
#define GAIN_KD 1.39         //ここを書き直す
#define GAIN_KI 349.0        //変更はプロフェッショナルコース
#define A_OFFSET 178.7       //ここを書き直す
/**********************************************/

//充電池の場合、以下の値に書き直す↑↑↑
//#define GAIN_KP 28.88
//#define GAIN_KD 0.69
//#define GAIN_KI 167.44

//AC電源の場合、以下の値に書き直す↑↑↑(注意、重量バランスの関係で電池はロボットに搭載してください。スイッチは入れない。)
//#define GAIN_KP 63.34
//#define GAIN_KD 1.25
//#define GAIN_KI 700.00

#define LOG_PAD 0
#define LOG_INPUT 0
#define MIN_ABS_SPEED 15     //モーターが動く最低の値

//充電池
//#define GAIN_KP 61.0
//#define GAIN_KD 2.81
//#define GAIN_KI 670.0

#define PID_SAMPLETIME 5     //5[msec]フィードバックループ

#define FWCRL_GAIN 10        //前後方向へのゲインリモートコントロール

//MPU
MPU6050 mpu;
PS2X ps2x;


// MPU control/status vars
bool dmpReady = false;
uint8_t mpuIntStatus;
uint8_t devStatus;
uint16_t packetSize;
uint16_t fifoCount;
uint8_t fifoBuffer[64];

// orientation/motion vars
Quaternion q;               // [w, x, y, z]         quaternion container
VectorFloat gravity;        // [x, y, z]            gravity vector
float ypr[3];               // [yaw, pitch, roll]   yaw/pitch/roll container and gravity vector


//PID


double kp, ki, kd;
double prevKp, prevKi, prevKd;
double originalSetpoint = A_OFFSET;;        //大きいと後ろへ
double setpoint = originalSetpoint;

double input, output;

PID pid(&input, &output, &setpoint, GAIN_KP, GAIN_KI, GAIN_KD, DIRECT);

//モータードライブIC設定
int ENA = 3;
int IN1 = 2;
int IN2 = 4;
int IN3 = 8;
int IN4 = 7;
int ENB = 6;

LMotorController motorController(ENA, IN1, IN2, ENB, IN3, IN4, 1, 1);

//タイマー用変数
long time1Hz = 0;
long time5Hz = 0;

//回数を数えるカウンター
long times = 0;

//-------------------------------------------
//関数
//-------------------------------------------

//--------------------------------------------
//PID Tuning (3 potentiometers)
// 比例、微分、積分パラメーターの調整を行う
//--------------------------------------------
void setPIDTuningValues(){
    {
        int potKp = analogRead(A0);
        int potKi = analogRead(A1);
        int potKd = analogRead(A2);

        kp = map(potKp, 0, 1023, 0, 15000) / 100.0;           //0 - 150
        ki = map(potKi, 0, 1023, 0, 100000) / 100.0;          //0 - 1000
        kd = map(potKd, 0, 1023, 0, 600) / 200.0;             //0 - 3


        if (kp != prevKp || ki != prevKi || kd != prevKd){
            Serial.print(kp);
            Serial.print(", ");
            Serial.print(ki);
            Serial.print(", ");
            Serial.print(kd);
            Serial.print(", ");
            Serial.println(input);
            pid.SetTunings(kp, ki, kd);
            prevKp = kp;
            prevKi = ki;
            prevKd = kd;
        }
    }
}

//--------------------------------------------
//ゼロ点と微分パラメーターの調整を行う
//--------------------------------------------
void setPointTunigValue(){
    int potZero = analogRead(A0);
    int potKd = analogRead(A1);

    originalSetpoint = 180 + (double)( 512 - potZero ) / 100; // from -5 to +5 deg
    kd = map(potKd, 0, 1023, 0, 600) / 200.0;                 //0 - 2
    ki = GAIN_KI;
    kp = GAIN_KP;

    if (kp != prevKp || ki != prevKi || kd != prevKd){
        Serial.print(kp);
        Serial.print(", ");
        Serial.print(ki);
        Serial.print(", ");
        Serial.print(kd);
        Serial.print(", ");
        Serial.println(originalSetpoint);

        pid.SetTunings(kp, ki, kd);
        prevKp = kp;
        prevKi = ki;
        prevKd = kd;
    }
}

//----------------------------
//1Hzで実行されるルーチン
//----------------------------
void loopAt1Hz(){
#ifdef MANUAL_PID_TUNING
    setPIDTuningValues();
#endif

#ifdef  MANUAL_ZERO_TUNING
    setPointTunigValue();
#endif
}

//----------------------------
//5Hzで実行されるルーチン
//----------------------------
void loopAt5Hz(){
}

//----------------------------
//割り込み関数
//----------------------------
volatile bool mpuInterrupt = false;        //DMP割り込み信号
void dmpDataReady(){                       //DMP割り込み信号を擬似的に作成(本来のハードウェア割り込みをソフトウェアで偽装する)
    mpuInterrupt = true;
}

//----------------------------
//初期設定
//----------------------------
void setup(){
    //コントローラーの設定
    ps2x.config_gamepad(13, 11, 10, 12, true, true);    //コントローラーを使うときのオマジナイ

    // I2Cバスを利用する
    Wire.begin();
    TWBR = 24;                                          // 400kHz I2C クロック

    Serial.begin(115200);                               //シリアル通信を使うときのオマジナイ
    while (!Serial);                                    // ウェイト

    // デバイス初期化
    Serial.println(F("Initializing I2C devices..."));
    mpu.initialize();

    // MPU通信確認
    Serial.println(F("Testing device connections..."));
    Serial.println(mpu.testConnection() ? F("MPU6050 connection successful") : F("MPU6050 connection failed"));

    // 初期設定DMP
    Serial.println(F("Initializing DMP..."));
    devStatus = mpu.dmpInitialize();

    // センサーオフセット
    mpu.setXGyroOffset(220);
    mpu.setYGyroOffset(76);
    mpu.setZGyroOffset(-85);
    mpu.setZAccelOffset(1688);

    // 動作確認
    if (devStatus == 0){
        // DMP起動
        Serial.println(F("Enabling DMP..."));
        mpu.setDMPEnabled(true);

        Serial.println(F("Enabling interrupt detection (Arduino external interrupt 0)..."));
        // タイマー１割り込みでおおよその検出をしてセンサーにデータを読み出す
        Timer1.initialize(10000);                       //10msec. update DMP per 100Hz
        Timer1.attachInterrupt(dmpDataReady);           //100Hzで起動する

        mpuIntStatus = mpu.getIntStatus();

        // DMPフラグ立てる
        Serial.println(F("DMP ready! Waiting for first interrupt..."));
        dmpReady = true;

        // パケットサイズを取得する
        packetSize = mpu.dmpGetFIFOPacketSize();

        //PID設定
        pid.SetMode(AUTOMATIC);
        pid.SetSampleTime(PID_SAMPLETIME);              //PID_SAMPLETIME msecサイクル
        pid.SetOutputLimits(-255, 255);
    }
    else{
        // 初期化エラー
        Serial.print(F("DMP Initialization failed (code "));
        Serial.print(devStatus);
        Serial.println(F(")"));
    }
}

//----------------------------
//シリアルプリント
//----------------------------
void pad_print(){
    Serial.print("Stick Values:");
    //  Serial.print(setpoint);
    Serial.print(100 * ((float)ps2x.Analog(PSS_RX) / 128 ));
    Serial.print(",");
    Serial.print(ps2x.Analog(PSS_LY) - 1, DEC);        //Left stick, Y axis. Other options: LX, RY, RX
    Serial.print(",");
    Serial.print(ps2x.Analog(PSS_LX), DEC);
    Serial.print(",");
    Serial.print(ps2x.Analog(PSS_RY) - 1, DEC);
    Serial.print(",");
    Serial.println(ps2x.Analog(PSS_RX), DEC);
}

//--------------------------------------------
//メインループ
//--------------------------------------------
void loop(){

    ps2x.read_gamepad();                                                                   //コントローラーから値を読み取る
    setpoint = originalSetpoint - (float)( ps2x.Analog(PSS_LY) - 1 ) * 0.001 * FWCRL_GAIN; //前後をコントロール

#if LOG_PAD
    pad_print();
#endif

    // MPUのデータの準備をまつ
    while (!mpuInterrupt && fifoCount < packetSize){
        pid.Compute();        //MPUのデータを待っている間にPID制御を行う

        //45度以上傾いた場合、モーターを止めて制御を行わない
        if (input < 135 || input > 225){
            motorController.move(0, 0);
        }
        else {
            motorController.move(output, MIN_ABS_SPEED);
        }

        //旋回ルーチン
        int lr_offset = ps2x.Analog(PSS_RX);
        times++;
        if (lr_offset != 0 && ( times % 50 ) == 0){
            motorController.move(-lr_offset, lr_offset, MIN_ABS_SPEED);
        }

        //現在の時刻を代入する
        unsigned long currentMillis = millis();
        if (currentMillis - time1Hz >= 1000){
            loopAt1Hz();
            time1Hz = currentMillis;
        }

        if (currentMillis - time5Hz >= 5000){
            loopAt5Hz();
            time5Hz = currentMillis;
        }
    }

    // 割り込みフラグをリセットする
    mpuInterrupt = false;
    mpuIntStatus = mpu.getIntStatus();

    // 現在のFIFOカウントを取得する
    fifoCount = mpu.getFIFOCount();

    //姿勢センサーからデータを取得する
    if (( mpuIntStatus & 0x10 ) || fifoCount == 1024){
        // オーバーフロー時にリセットする
        mpu.resetFIFO();
        Serial.println(F("FIFO overflow!"));
    }                                             //DMP割り込み状況を確認する
    else
    if (mpuIntStatus & 0x02)
    {
        // 正しい長さのデータを取得するまで待つ
        while (fifoCount < packetSize) fifoCount = mpu.getFIFOCount();

        // パケット受信する
        mpu.getFIFOBytes(fifoBuffer, packetSize);
        fifoCount -= packetSize;
        //データを角度へ変換する
        mpu.dmpGetQuaternion(&q, fifoBuffer);
        mpu.dmpGetGravity(&gravity, &q);
        mpu.dmpGetYawPitchRoll(ypr, &q, &gravity);

#if LOG_INPUT
        Serial.print("ypr\t");
        Serial.print(ypr[0] * 180 / M_PI);
        Serial.print("\t");
        Serial.print(ypr[1] * 180 / M_PI);
        Serial.print("\t");
        Serial.println(ypr[2] * 180 / M_PI);
#endif
        input = ypr[2] * 180 / M_PI + 180;        //角度を計算して、PIDの制御入力にする
    }
}
