//-----------------------------------------------------------------------
// robot.cpp
// ロボット関係　モーター　センサー　等
// ROBOPRO RobotBattler
//  written by kodachi kodachi@furo  2014/6/16
//-----------------------------------------------------------------------
// Copyright (C) Future Robotics Technology Center All Right Reserved
//-----------------------------------------------------------------------
#include <Arduino.h>
#include <RPlib.h>
#include "command.h"
#include "robot.h"
#include "input.h"

extern InputManager *getInput();

extern int32_t GetDefaultMutekiTime(); // メインファイルに。

RPmotor mcl(MC2);                      //MC2に繋がっているモータを指定する(左脚)
RPmotor mcr(MC1);                      //MC1に繋がっているモータを指定する(右脚)

inline float SIGNF(float A){
    return ( A >= 0.0f ? 1.0f : -1.0f );
}

void Robot::setup(){
    // TouchSensor
    pinMode(A2, INPUT_PULLUP);         //  おまじない，ピン設定（コネクタA2)
    pinMode(A3, INPUT_PULLUP);         //  おまじない，ピン設定（コネクタA3)
    pinMode(A4, INPUT_PULLUP);         //  おまじない，ピン設定（コネクタA4)
//	pinMode(A5,INPUT_PULLUP); //  おまじない，ピン設定（コネクタA5)

    init();
}

void Robot::update(){
    // 通常時
    float ll, rr;
    PS2X *ps = &getInput()->ps2x;

    ll = ps->Analog(PSS_LY);        //アナログ右スティック上下
    rr = ps->Analog(PSS_RY);        //アナログ左スティック上下

    ll = constrain(ll, -127, 127);  // llの値が-127から127の間におさまるようにする
    rr = constrain(rr, -127, 127);  // rrの値が-127から127の間におさまるようにする

    if(isAlive()){
        // Robotにスピード値を設定
        setSpeedL(ll);
        setSpeedR(rr);
    }

    // センサー情報更新
    // タッチセンサーの状態を判断する  HIGH or LOW
    bool wa;
    bool tmp;

    tmp                    = m_touch[TOUCH2].repeat; // 1フレ前
    m_touch[TOUCH2].repeat = digitalRead(A2);
    wa                     = tmp ^ m_touch[TOUCH2].repeat;
    m_touch[TOUCH2].trg    = wa & m_touch[TOUCH2].repeat;
    m_touch[TOUCH2].rTrg   = wa & tmp;

    tmp                    = m_touch[TOUCH3].repeat; // 1フレ前
    m_touch[TOUCH3].repeat = digitalRead(A3);
    wa                     = tmp ^ m_touch[TOUCH3].repeat;
    m_touch[TOUCH3].trg    = wa & m_touch[TOUCH3].repeat;
    m_touch[TOUCH3].rTrg   = wa & tmp;

    tmp                    = m_touch[TOUCH4].repeat; // 1フレ前
    m_touch[TOUCH4].repeat = digitalRead(A4);
    wa                     = tmp ^ m_touch[TOUCH4].repeat;
    m_touch[TOUCH4].trg    = wa & m_touch[TOUCH4].repeat;
    m_touch[TOUCH4].rTrg   = wa & tmp;

#if 0
    tmp                    = m_touch[TOUCH5].repeat; // 1フレ前
    m_touch[TOUCH5].repeat = digitalRead(A5);
    wa                     = tmp ^ m_touch[TOUCH5].repeat;
    m_touch[TOUCH5].trg    = wa & m_touch[TOUCH5].repeat;
    m_touch[TOUCH5].rTrg   = wa & tmp;
#endif
}

bool Robot::checkCommand(int8_t cmd){
    bool ret = false;
    switch(cmd){
    case COMMAND_NONE:
    default:
        break;

    case COMMAND_ROBOT_LIFE_DEC:
        if(!isSpecial() && isAlive()){
            decLife();        // スペシャル中は無敵
        }
        break;

    case COMMAND_ROBOT_STAMINA_ADD:
        if(!isSpecial() && isAlive()){
            addStamina();
        }
        break;

    case COMMAND_ROBOT_INIT:
        init();
        break;

    case COMMAND_ROBOT_DEATH:
        death();        // 瞬殺
        break;

    case COMMAND_ROBOT_SPECIAL:
        if(!isSpecial() && m_stayTimer == 0){
            setSpecial();
        }
        break;

    case COMMAND_ROBOT_STAMINA_MAX:
        spStaminaRecover();
        break;
    }

    return ret;
}

void Robot::setSpecial(){
    if(m_stamina >= 2) m_stamina -= 2;
    else return;

    m_step = STEP_SPECIAL;
    //	m_spTimer=100;

    if(GetDefaultMutekiTime() < 100)
        m_spTimer = GetDefaultMutekiTime();
    else
        m_spTimer = 100;
}

// lifeを1使って、スタミナを全快する
void Robot::spStaminaRecover(){
    if(m_life > 1 && m_stamina != STAMINA_MAX){
        m_life--;
        m_stamina = STAMINA_MAX;
    }
}

int8_t Robot::exec(){
    int8_t cmd   = 0;
    float speedL = getSpeedL();
    float speedR = getSpeedR();

    // スピードからスタミナを減らす処理
    // ※ abs(): Arduino 実装の都合によりカッコ内で関数を使ったり変数を操作することができない
    if(abs(speedL) + abs(speedR) > 200.0f
       || abs(speedL) > 100.0f
       || abs(speedR) > 100.0f
       )
    {
        if(m_runtimer > 0){
            m_runtimer--;
        }
        else{
            if(m_stamina > 0){
                m_stamina--;
                m_runtimer = m_stamina * ( m_stamina2 * 2 );        // m_stamina*8
            }
        }
    }
    else{
        if(abs(speedL) + abs(speedR) < 128.0f){
            if(m_runtimer < RUNTIMER_MAX) m_runtimer++;
        }
    }

    // ロボット実行とロボットからのメッセージ取得	exec_main
    switch(getStep()){
    case STEP_INIT:        // 初期化ステップ
        setState(STATE_ALIVE);
        setStep(STEP_STAY);
        break;

    default:
    case STEP_STAY:
        stay();
        break;

    case STEP_WALK:
        walk();
        break;

    case STEP_RUN:
        run();
        break;

    case STEP_SPECIAL:
        special();
        break;
    }

    return cmd;
}

void Robot::stay(){
    float speedL = getSpeedL();
    float speedR = getSpeedR();

    mcl.rotate(0);        //左のモータを回転(-255〜255の範囲で指定)
    mcr.rotate(0);        //右のモータを回転(-255〜255の範囲で指定)

    if(m_stayTimer > 0) m_stayTimer--;
    if(m_stayTimer == 0){
        if(abs(speedL) > 10.0f || abs(speedR) > 10.0f){
            if(getStamina() > 4)
                setStep(STEP_RUN);
            else if(getStamina() >= 1)
                setStep(STEP_WALK);
        }
    }
}

void Robot::run(){
    float speedL = getSpeedL();
    float speedR = getSpeedR();

    mcl.rotate(-2 * speedL);       //左のモータを回転(-255〜255の範囲で指定)
    mcr.rotate(2 * speedR);        //右のモータを回転(-255〜255の範囲で指定)

    if(abs(speedL) + abs(speedR) < 10.0f){
        setStep(STEP_STAY);
    }
    else if(abs(speedL) + abs(speedR) < 100.0f){
        setStep(STEP_WALK);
    }
    else if(getStamina() <= 0){
        setStep(STEP_STAY);
    }
    else if(getStamina() <= 4){
        setStep(STEP_WALK);
    }
}

void Robot::walk(){
    float speedL = getSpeedL();              //-127 -- 127
    float speedR = getSpeedR();

    // 上限をつける
    if(abs(speedL) > 80.0f){
        mcl.rotate(-2 * SIGNF(speedL) * 80); //左のモータを回転(-255〜255の範囲で指定)
    }
    else{
        mcl.rotate(-2 * speedL);             //左のモータを回転(-255〜255の範囲で指定)
    }
    if(abs(speedR) > 80.0f){
        mcr.rotate(2 * SIGNF(speedR) * 80);  //右のモータを回転(-255〜255の範囲で指定)
    }
    else{
        mcr.rotate(2 * speedR);              //右のモータを回転(-255〜255の範囲で指定)
    }

    if(abs(speedL) + abs(speedR) < 10.0f){
        setStep(STEP_STAY);
    }
    else if(getStamina() <= 0){
        setStep(STEP_STAY);
    }
    else if(abs(speedL) + abs(speedR) > 200.0f){
        if(getStamina() > 4)
            setStep(STEP_RUN);
    }
}

void Robot::special(){
    // ぐるぐる
    mcl.rotate(255);        //左のモータを回転(-255〜255の範囲で指定)
    mcr.rotate(255);        //右のモータを回転(-255〜255の範囲で指定)

    if(m_spTimer){
        m_spTimer--;
    }

    if(m_spTimer == 0){
        setStep(STEP_STAY);
        m_stayTimer = 20;
    }
}

void Robot::setStatus(int8_t life, int8_t stamina, int8_t recover){
    if(life + stamina + recover > 10
       || life <= 0
       || stamina <= 0
       || recover <= 0)
    {
        m_life     = 0;
        m_recover  = 1;
        m_stamina2 = 1;
        return;
    }

    m_life     = life;
    m_stamina2 = stamina;
    m_recover  = recover;
}
