//------------------------------------------------------------------------
//ヨー・ピッチ・ロール（ypr）計測を用いたプログラム
//奥／手前方向への回転角度をリアルタイムで測り、角度に応じてスピーカーからの周波数を変更。
//※計測した角度はypr[2]に記録された後、換算されて変数「input」に格納されている
//スピーカーをD2に接続。シリアルモニタの設定は「115200baud」
//-------------------------------------------------------------------------

#include <PID_v1.h>
#include <TimerOne.h>
//#include "LMotorController.h"
#include "I2Cdev.h"
//#include <PS2X_lib.h>
#include "MPU6050_6Axis_MotionApps20R.h"
#include "Wire.h"
#include <Tone.h>
#include <RPlib.h>

#define MANUAL_ZERO_TUNING   //ゼロ点調整と　微分項の調整(D) (上記と排他)

#define GAIN_KP 39.0
#define GAIN_KD 1.39
#define GAIN_KI 349.0
#define A_OFFSET 178.7


#define LOG_PAD 0
#define LOG_INPUT 0
#define MIN_ABS_SPEED 15     //モーターが動く最低の値

#define PID_SAMPLETIME 5     //5[msec]フィードバックループ

#define FWCRL_GAIN 10        //前後方向へのゲインリモートコントロール

//MPU
MPU6050 mpu;

Tone spk;

// MPU control/status vars
bool dmpReady = false;
uint8_t mpuIntStatus;
uint8_t devStatus;
uint16_t packetSize;
uint16_t fifoCount;
uint8_t fifoBuffer[64];

// orientation/motion vars
Quaternion q;               // [w, x, y, z]         quaternion container
VectorFloat gravity;        // [x, y, z]            gravity vector
float ypr[3];               // [yaw, pitch, roll]   yaw/pitch/roll container and gravity vector

double kp, ki, kd;
double prevKp, prevKi, prevKd;
double originalSetpoint = A_OFFSET;;        //大きいと後ろへ
double setpoint = originalSetpoint;

double input, output;

PID pid(&input, &output, &setpoint, GAIN_KP, GAIN_KI, GAIN_KD, DIRECT);

//タイマー用変数
long time1Hz = 0;
long time5Hz = 0;

//回数を数えるカウンター
long times = 0;

//-------------------------------------------
//関数
//-------------------------------------------

//----------------------------
//1Hzで実行されるルーチン
//----------------------------
void loopAt1Hz(){
#ifdef MANUAL_PID_TUNING
    setPIDTuningValues();
#endif
}
//----------------------------
//5Hzで実行されるルーチン
//----------------------------
void loopAt5Hz(){
}

//----------------------------
//割り込み関数
//----------------------------
volatile bool mpuInterrupt = false;        //DMP割り込み信号
void dmpDataReady(){                       //DMP割り込み信号を擬似的に作成(本来のハードウェア割り込みをソフトウェアで偽装する)
    mpuInterrupt = true;
}

//----------------------------
//初期設定
//----------------------------
void setup(){

    // I2Cバスを利用する
    Wire.begin();
    TWBR = 24;                                          // 400kHz I2C クロック

    Serial.begin(115200);                               //シリアル通信を使うときのオマジナイ
    while (!Serial);                                    // ウェイト

    // デバイス初期化
    Serial.println(F("Initializing I2C devices..."));
    mpu.initialize();

    // MPU通信確認
    Serial.println(F("Testing device connections..."));
    Serial.println(mpu.testConnection() ? F("MPU6050 connection successful") : F("MPU6050 connection failed"));

    // 初期設定DMP
    Serial.println(F("Initializing DMP..."));
    devStatus = mpu.dmpInitialize();

    // センサーオフセット
    mpu.setXGyroOffset(220);
    mpu.setYGyroOffset(76);
    mpu.setZGyroOffset(-85);
    mpu.setZAccelOffset(1688);

    // 動作確認
    if (devStatus == 0){
        // DMP起動
        Serial.println(F("Enabling DMP..."));
        mpu.setDMPEnabled(true);

        Serial.println(F("Enabling interrupt detection (Arduino external interrupt 0)..."));
        // タイマー１割り込みでおおよその検出をしてセンサーにデータを読み出す
        Timer1.initialize(10000);                       //10msec. update DMP per 100Hz
        Timer1.attachInterrupt(dmpDataReady);           //100Hzで起動する

        mpuIntStatus = mpu.getIntStatus();

        // DMPフラグ立てる
        Serial.println(F("DMP ready! Waiting for first interrupt..."));
        dmpReady = true;

        // パケットサイズを取得する
        packetSize = mpu.dmpGetFIFOPacketSize();

        //PID設定
        pid.SetMode(AUTOMATIC);
        pid.SetSampleTime(PID_SAMPLETIME);              //PID_SAMPLETIME msecサイクル
        pid.SetOutputLimits(-255, 255);
    }
    else{
        // 初期化エラー
        Serial.print(F("DMP Initialization failed (code "));
        Serial.print(devStatus);
        Serial.println(F(")"));
    }
    
    spk.begin(D2);
}

//----------------------------
//シリアルプリント
//----------------------------
void pad_print(){
}

int freq;
//--------------------------------------------
//メインループ
//--------------------------------------------
void loop(){
    // MPUのデータの準備をまつ
    while (!mpuInterrupt && fifoCount < packetSize){
        pid.Compute();        //MPUのデータを待っている間にPID制御を行う
        
        //現在の時刻を代入する
        unsigned long currentMillis = millis();
        if (currentMillis - time1Hz >= 1000){
            loopAt1Hz();
            time1Hz = currentMillis;
        }

        if (currentMillis - time5Hz >= 5000){
            loopAt5Hz();
            time5Hz = currentMillis;
        }
    }

    // 割り込みフラグをリセットする
    mpuInterrupt = false;
    mpuIntStatus = mpu.getIntStatus();

    // 現在のFIFOカウントを取得する
    fifoCount = mpu.getFIFOCount();

    //姿勢センサーからデータを取得する
    if (( mpuIntStatus & 0x10 ) || fifoCount == 1024){
        // オーバーフロー時にリセットする
        mpu.resetFIFO();
        Serial.println(F("FIFO overflow!"));
    }                                             //DMP割り込み状況を確認する
    else
    if (mpuIntStatus & 0x02)
    {
        // 正しい長さのデータを取得するまで待つ
        while (fifoCount < packetSize) fifoCount = mpu.getFIFOCount();

        // パケット受信する
        mpu.getFIFOBytes(fifoBuffer, packetSize);
        fifoCount -= packetSize;
        //データを角度へ変換する
        mpu.dmpGetQuaternion(&q, fifoBuffer);
        mpu.dmpGetGravity(&gravity, &q);
        mpu.dmpGetYawPitchRoll(ypr, &q, &gravity);
        input = ypr[2] * 180 / M_PI + 180;        //角度を計算して、PIDの制御入力にする
    }
    
    Serial.println(input);
    freq = constrain(input, 90, 270);  //最大値・最小値はinputの値を見ながら調整する（直立状態の時の値がちょうど中間あたりに来るように）
    freq = map(freq, 90, 270, 262, 523);  //角度の値を周波数の値に換算しなおす（262Hz～523Hzでほぼ「ドレミファソラシド」1オクターブ分になる）
    spk.play(freq, 500);
    
}
